//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12
import QtQuick.Window 2.12

import wesual.Ui 1.0

import wesual.Ui.Private 1.0

FocusScope {
    id : uiDropDown

    // - Data state
    property int currentIndex : 0
    readonly property var currentElement : {
        if (!model || currentIndex < 0)
            return null;

        try {
            if (model.length) {
                // JS array or list property
                return model[currentIndex];
            } else if (model.get) {
                // List model
                return model.get(currentIndex);
            }
        } catch (e) {
            return null;
        }
    }

    property bool bindable    : false

    // - Interaction state
    readonly property alias hovered : mouseArea.containsMouse
    readonly property bool  pressed :
        mouseArea.containsPress || popupAnchor.popupVisible

    // - Delegates
    property Component delegate
    property Component highlightDelegate
    property Component labelDelegate
    property Component backgroundDelegate : UiDropDownBackground {
        anchors.fill : uiDropDown
    }

    property int leftMargin   : 6
    property int rightMargin  : 24
    property int topMargin    : 4
    property int bottomMargin : 4

    // - Popup properties
    property int popupColumns           : 1
    property int popupVerticalSpacing   : 0
    property int popupHorizontalSpacing : 0

    property int popupLeftMargin   : 0
    property int popupRightMargin  : 0
    property int popupTopMargin    : 3
    property int popupBottomMargin : 3

    // Not functional any more - kept for compatibility
    property bool sizeFollowsPopup : false

    // - Model
    property var model : []
    readonly property int count : {
        if (!model) {
            return 0;
        }

        if ("length" in model)
            return model.length;
        else if ("count" in model)
            return model.count;

        return 0;
    }

    // Removed functionality, kept property for compatibility reasons
    property bool hideCurrent : false

    signal indexPicked(int index)

    implicitWidth    : p_.implicitWidth
    implicitHeight   : p_.implicitHeight
    activeFocusOnTab : true

    baselineOffset   : p_.label ? p_.label.baselineOffset + topMargin : 0

    // Key handling
    Keys.forwardTo  : popupAnchor.popupVisible ? popupAnchor.popupItem : null
    Keys.onSpacePressed : popupAnchor.show()
    Keys.onUpPressed    : {
        var index = Math.max(currentIndex - 1, 0);
        if (index != currentIndex) {
            if (!uiDropDown.bindable) {
                currentIndex = index;
            }
            uiDropDown.indexPicked(index);
        }
    }
    Keys.onDownPressed  : {
        var index = Math.min(currentIndex + 1, count - 1);
        if (index != currentIndex) {
            if (!uiDropDown.bindable) {
                currentIndex = index;
            }
            uiDropDown.indexPicked(index);
        }
    }

    Component.onCompleted : {
        if (backgroundDelegate) {
            var item = backgroundDelegate.createObject(uiDropDown);
            item.z = -1;
            p_.background = item;
        }
        if (labelDelegate) {
            p_.label = labelDelegate.createObject(uiDropDown);
            p_.label.x = leftMargin;
            p_.label.y = topMargin;
            p_.label.width = Qt.binding(
                        () => uiDropDown.width - leftMargin - rightMargin);
        }
    }

    QtObject {
        id : p_

        readonly property int implicitWidth : {
            return (label ? label.implicitWidth : 40) +
                uiDropDown.leftMargin + uiDropDown.rightMargin;
        }
        readonly property int implicitHeight :
            (label ? label.implicitHeight : 16) +
            uiDropDown.topMargin + uiDropDown.bottomMargin

        property Item label
        property Item background

        property int implicitPopupWidth
    }

    MouseArea {
        id : mouseArea

        anchors.fill : uiDropDown
        hoverEnabled : true
        onPressed    : popupAnchor.show()
        onReleased   : {
            var popupPos = mapToItem(popupAnchor.popupItem, mouseX, mouseY);
            if (popupAnchor.popupItem.contains(popupPos)) {
                popupAnchor.popupItem.pickItemAt(
                            Qt.point(mouseX, mouseY), mouseArea);
            }
        }
    }

    UiPopupAnchor {
        id : popupAnchor

        parent : p_.background ? p_.background : uiDropDown

        anchors {
            bottom       : parent.bottom
            bottomMargin : 1
            left         : parent.left
            topMargin    : 0
        }
        popupCreationPolicy : UiPopupAnchor.OnDemandCreation

        triggerItem : mouseArea
        autoLayout  : true
        persistent  : false

        onAboutToHide : popupItem.reset()

        popup : Rectangle {
            id : popup

            property var itemList

            function reset() {
                scrollView.scrollTo(0);
            }

            function pickItemAt(pos, container) {
                pos = content.mapFromItem(container, pos.x, pos.y);
                var item = content.childAt(pos.x, pos.y);
                if (item) {
                    var index = popup.itemList.indexOf(item);
                    if (index !== -1) {
                        if (!uiDropDown.bindable) {
                            uiDropDown.currentIndex = index;
                        }
                        uiDropDown.indexPicked(index);
                    }
                    popupAnchor.hide();
                }
            }

            function updateItemList() {
                var items = [];

                for (var i = 0; i < repeater.count; ++i) {
                    var item = repeater.itemAt(i);
                    if (item) {
                        items.push(item);
                    }
                }

                itemList = items;
            }

            readonly property Item highlightItem : {
                var hit = null;
                if (mouseArea.pressed) {
                    var pos = mouseArea.mapToItem(
                                content, mouseArea.mouseX, mouseArea.mouseY);
                    hit = content.childAt(pos.x, pos.y);
                } else if (popupMouseArea.containsMouse) {
                    pos = popupMouseArea.mapToItem(
                                content, popupMouseArea.mouseX,
                                popupMouseArea.mouseY);
                    hit = content.childAt(pos.x, pos.y);
                } else if (currentIndex !== -1) {
                    hit = repeater.itemAt(currentIndex);
                }

                return hit;
            }

            implicitWidth   : {
                const ww = uiDropDown.Window.width;
                const avail = Math.max(
                                ww - popupAnchor.distanceLeft,
                                ww - popupAnchor.distanceRight);
                const max = avail - 50;
                const pw = popupAnchor.parent.width;
                var w = Math.max(Math.min(max, p_.implicitPopupWidth), pw);

                if (scrollView.heightRatio < 1.0) {
                    w += 17;
                }

                return w;
            }
            implicitHeight  : content.implicitHeight

            color   : UiColors.getColor(UiColors.White)
            opacity : 0
            enabled : popupAnchor.popupVisible
            border {
                color : UiColors.getColor(UiColors.ControlGrey20)
                width : 1
            }

            states : [
                State {
                    name : "visible"
                    when : popupAnchor.popupVisible

                    PropertyChanges {
                        target  : popup
                        opacity : 1
                    }
                }
            ]

            // Key handling
            Keys.onSpacePressed : popupAnchor.hide()
            Keys.onUpPressed    : {}
            Keys.onDownPressed  : {}
            Keys.onLeftPressed  : {}
            Keys.onRightPressed : {}

            Item {
                id : highlight

                opacity : popup.highlightItem ? 1 : 0
                parent  : popup.highlightItem
                z       : -1

                anchors.fill : parent || null

                Component.onCompleted : {
                    if (uiDropDown.highlightDelegate) {
                        const item = uiDropDown.highlightDelegate.createObject(
                                       highlight);
                        item.anchors.fill = highlight;
                    }
                }
            }

            MouseArea {
                id : popupMouseArea

                anchors {
                    top    : popup.top
                    left   : popup.left
                    bottom : popup.bottom
                }
                width : content.width

                hoverEnabled : true
                onReleased   : {
                    popup.pickItemAt(Qt.point(mouseX, mouseY), popupMouseArea);
                }
            }

            UiVerticalScrollView {
                id : scrollView

                anchors.fill    : parent
                scrollbarOffset : 2

                Item {
                    id : content

                    implicitWidth  : 10
                    implicitHeight : 10

                    property bool initialized : false

                    function update() {
                        Qt.callLater(performLayout)
                    }

                    function performLayout() {
                        var x = popupLeftMargin;
                        var y = popupTopMargin;

                        var columns = uiDropDown.popupColumns;
                        var maxX = 0;
                        var maxY = 0;
                        for (var i = 0; i < repeater.count; ++i) {
                            var item = repeater.itemAt(i);
                            if (item) {
                                item.visible = true;
                                item.x = x;
                                item.y = y;

                                maxX = Math.max(maxX, x + item.implicitWidth);
                                maxY = Math.max(maxY, y + item.implicitHeight);
                                x += item.implicitWidth +
                                        popupHorizontalSpacing;
                                columns--;
                                if (columns <= 0) {
                                    columns = uiDropDown.popupColumns;
                                    y += item.implicitHeight +
                                            popupVerticalSpacing;
                                    x = popupLeftMargin;
                                }
                            }
                        }
                        content.implicitWidth  = maxX + popupRightMargin;
                        content.implicitHeight = maxY + popupBottomMargin;

                        p_.implicitPopupWidth = content.implicitWidth;
                    }

                    Repeater {
                        id : repeater

                        model       : uiDropDown.model
                        delegate    : uiDropDown.delegate
                        onItemAdded : {
                            popup.updateItemList();
                            if ("highlight" in item) {
                                item.highlight = Qt.binding(function() {
                                    return this === popup.highlightItem;
                                });
                            }
                            item.onImplicitWidthChanged.connect(
                                        content.update);
                            item.onImplicitHeightChanged.connect(
                                        content.update);

                            if (!content.initialized) {
                                content.initialized = true;
                                content.performLayout();
                            } else {
                                content.update();
                            }
                        }
                        onItemRemoved : {
                            popup.updateItemList();
                            content.update();
                        }
                    }
                }
            }

            Connections {
                target : uiDropDown
                onCurrentIndexChanged : content.update()
            }
        }
    }
}
